/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_13; import java.util.*; import lotus.domino.*;

/**
 * Ein DJCacheDocument ist ein Nachbau eines lotus.domino.Document und verhlt
 * sich weitestgehend wie dieses, insbesondere fr die Methoden getItemValueString,
 * getItemValueInteger und getItemValueDouble.
 * Es basiert lediglich auf Java Mitteln und ist unabhngig von einer Domino Session.
 * Hierdurch wird es mglich ein solches Document speicherresident (z.B. durch einen
 * Singleton) vorzuhalten. Dies ist insbesondere fr hufig bentigte
 * Konfigurationsdokumente interessant oder fr Dokumente, deren Erzeugung teuer ist.
 * Durch die Nachahmung der Mimik des lotus.domino.Document kann in Anwendungen
 * sehr leicht das lotus.domino.Document durch ein DJCacheDocument ausgetauscht werden.
 * 
 * Allerdings werden keine Sicherheitsfunktionen von Domino implementiert.
 * Insbesondere werden verschlsselte Felder eines Domino Document unverschlsselt 
 * gespeichert.
 * @author Thomas ekert
 * @see lotus.domino.Document
 */
public class DJCacheDocument {
	
	private static final int DJCACHEDOMINOEXCEPTIONID = 999;
	private static final boolean SIMULATE_R7 = true;
	private static final boolean IS_IIOP = false;
	private Map elements = Collections.synchronizedMap(new HashMap ());
	private FieldNames fn = new FieldNames();
	
	private String universalID = null;
	private String parentDocumentUNID = null;
	private String parentDatabaseFilePath = null;
	private String parentDatabaseFileName = null;
	private String parentDatabaseReplicaID = null;

	/**
	 * Erstellt aus einem lotus.domino.Document ein DJCacheDocument
	 * @param doc
	 */
	public DJCacheDocument(Document doc) throws NotesException {
		if (doc != null) {
			init (doc);
		}
	}

	/**
	 * Fgt dem Feld (fieldname) einen zustzlichen Wert (value) hinzu. Existiert das Feld nicht, wird es erstellt.
	 * Befinden sich in einem Vector DateTime Objekte, werden diese als java.util.Date abgespeichert.
	 * Intern werden alle Elemente als Vector gespeichert, um die Mimik von lotus.domino.Document nachzuahmen.
	 * Besondere Eigenschaften, wie isAuthors oder isReaders werden zur Zeit nicht untersttzt.
	 * @param fieldname
	 * @param value
	 */
	public void appendItemValue(String fieldName, Object value) {
		if (!ok (fieldName)) {
			return;
		}
		if (value == null) {
			value = "";
		}
		Vector v = new Vector();
		boolean isDateTime = false;
		if (value instanceof Vector) {
			v = (Vector)value;
			if (isInteger(v)) {
				v = toDouble(v);
			} else if (isDateTime(v)) {
				isDateTime=true;
				v = toJavaDate(v);
			}
		} else if (value instanceof Integer) {
			v.addElement(new Double(((Integer)value).doubleValue()));
		} else if (value instanceof DateTime) {
			isDateTime=true;
			try {
				v.addElement (((DateTime) value).toJavaDate());
			} catch (NotesException e) {
				v = new Vector();
				v.addElement ("");
			}
		} else {
			v.addElement(value);
		}
		elements.put(fn.addName(fieldName), v);
		if (isDateTime) {
			fn.setType(fieldName, DateTime.class.getName());
		}
	}

	/**
	 * Erstellt ein Feld (feldname) falls es nicht existiert.
	 * @param fieldName
	 * @param value
	 */
	public void appendItemValue(String fieldName) {
		appendItemValue(fieldName, "");
	}

	/**
	 * Fgt dem Feld (fieldname) einen zustzlichen Wert (value) hinzu. Existiert das Feld nicht, wird es erstellt.
	 * ACHTUNG Notes Speichert intern Integer Variablen als Double.
	 * @param fieldName
	 * @param value
	 */
	public void appendItemValue(String fieldName, int value) {
		appendItemValue(fieldName, new Double(value));
	}

	/**
	 * Fgt dem Feld (fieldname) einen zustzlichen Wert (value) hinzu. Existiert das Feld nicht, wird es erstellt.
	 * @param fieldName
	 * @param value
	 */
	public void appendItemValue(String fieldName, double value) {
		appendItemValue(fieldName, new Double(value));
	}

	/**
	 * Liefert den Inhalt eines Feldes (fieldname) das mehrere Werte enthlt oder einen
	 * einzelnen Wert als Element eines neuen Vectors.
	 * Intern werden Werte nur als Vector gespeichert.
	 * @see appendItemValue
	 * @param fieldname
	 * @return
	 */
	public Vector getItemValue(String fieldName) {
		if (!hasItem (fieldName)) {
			return new Vector();
		}
		Object result = elements.get(fn.makeName(fieldName));
		if (result != null) {//Werte werden intern nur als Vector gespeichert
			return (Vector) result;
		} else {
			return new Vector();
		}
	}

	/**
	 * Liefert den Inhalt eine Feldes (fieldname) als String.
	 * @param fieldname
	 * @return
	 */
	public String getItemValueString(String fieldName) {
		if (!hasItem (fieldName)) {
			return IS_IIOP?null:SIMULATE_R7?"":null;
		}
		Vector retVal = getItemValue (fieldName);
		try {
			return (String) retVal.firstElement();
		} catch (Exception otherException) {
			return SIMULATE_R7?"":null; // sonst null/"".
		}
	}

	/**
	 * Liefert den Inhalt eine Feldes (fieldname) als int.
	 * Imitiert dabei das Round Verhalten von Domino. (round (-1.5) = -2))
	 * @param fieldname
	 * @return
	 */
	public int getItemValueInteger(String fieldName) {
		if (!hasItem (fieldName)) {
			return 0;
		}
		double val = getItemValueDouble (fieldName);
		if (!IS_IIOP) {
			int mult = val < 0?-1:1;
			return mult * Math.round(new Double (Math.abs (val)).floatValue());
		} else {
			return Math.round(new Float(val).floatValue());
		}
	}

	/**
	 * Liefert den Inhalt eine Feldes (fieldname) als double.
	 * @param fieldname
	 * @return
	 */
	public double getItemValueDouble(String fieldName) {
		if (!hasItem (fieldName)) {
			return 0.0;
		}
		Vector retVal = getItemValue (fieldName);
		try {
			return ((Double) retVal.firstElement()).doubleValue();
		} catch (Exception e) {
			return 0.0;
		}
	}
	
	/**
	 * Ldt einen gespeicherten Vector und versucht die enthaltenen Werte in DateTime umzuwandeln.
	 * Hierfr wird eine Session bentigt, um ein DateTime Objekt zu erzeugen, daher der
	 * zustzliche Parameter.
	 * Alternativ kann getItemValue verwendet werden, dann wird ein Vector mit Date Elementen
	 * zurckgegeben.
	 * Beim Speichern von DateTime Items wird immer sowohl Datum als auch Zeit gespeichert.
	 * Reine Datums- (oder Zeit-) Items erhalten daher, anders als in lotus.domino.Document
	 * immer beide Komponenten (wg. der Umwandlung zwischen DateTime und Date).
	 * @param fieldName
	 * @param session
	 * @return
	 * @throws NotesException - falls das Item nicht vom DateTime Typ ist
	 */
	public Vector getItemValueDateTimeArray (String fieldName, Session session) throws NotesException {
		String type = fn.getType(fieldName);
		if (DateTime.class.getName().equals(type)) {
			Vector source = getItemValue(fieldName);
			Vector result = new Vector();
			for (int i = 0; i<source.size();i++){
				result.add (session.createDateTime((Date)source.elementAt(i)));
			}
			return result;
		} else {
			if (!IS_IIOP) {
			throw new NotesException (NotesError.NOTES_ERR_NOT_A_DATE_ITEM,"Item value is not a date type");
			} else {
				return new Vector();
			}
		}
	}
	
	/**
	 * Liefert den Typ fr ein Item.
	 * Zur Zeit nur fr DateTime Items implementiert. Ansonsten wird null zurckgegeben.
	 * @param fieldName
	 * @return
	 */
	public String getType (String fieldName) {
		return fn.getType(fieldName);
	}
	
	/**
	 * Gibt alle Item Namen des DJCacheDocument zurck.
	 * Exisitieren doppelte Items mit gleichem Namen, wird ein solcher Name
	 * lediglich einmal zurckgegeben.
	 * @return
	 */
	public Vector getAllItemNames () {
		return fn.getAllNames();
	}
	
	/**
	 * Liefert die DocumentUNID eines optional vorhandenen Mutterdokuments
	 * desjenigen Document aus dem das DJCacheDocument ursprnglich erzeugt wurde.
	 * @return
	 */
	public String getParentDocumentUNID() {
		return parentDocumentUNID;
	}

	/**
	 * Liefert die DocumentUNID des Document aus dem das DJCacheDocument ursprnglich
	 * erzeugt wurde.
	 * @return
	 */
	public String getUniversalID() {
		return universalID;
	}


	/**
	 * Ersetzt den Ihnalt des Feldes (fieldname) durch den Wert (value). Existiert das Feld nicht, wird es erstellt.
	 * @param fieldname
	 * @param value
	 */
	public void replaceItemValue(String fieldName, Object value) {
		removeItem (fieldName);
		appendItemValue (fieldName, value);
	}
	
	/**
	 * Entfernt alle Items mit dem Namen fieldName
	 * @param fieldName
	 */
	public void removeItem (String fieldName) {
		if (!fn.hasName(fieldName)) {
			return;
		}
		Vector v = fn.getNames(fieldName);
		for (int i = 0; i < v.size(); i++) {
			elements.remove ((String) v.elementAt(i));
		}
		fn.removeName(fieldName);
	}

	/**
	 * berprft ob es das Feld (fieldname) gibt.
	 * @param fieldname
	 * @return
	 */
	public boolean hasItem(String fieldName) {
		return fn.hasName(fieldName);
	}

	/**
	 * Die recycle Methode ist nur der Vollstndigkeit (Mimik von lotus.domino.Document)
	 * halber vorhanden, muss aber keine Aufgaben erledigen.
	 */
	public void recycle() {
	}

	/**
	 * Gibt den Kompletten Inhalt des Dokuments als String zurck.
	 * Einzelne Felder werden durch kommatas getrennt.
	 */
	public String toString() {
		String tmp = elements.toString().replace(',', '\n');
		return " " + tmp.substring(1, tmp.length() - 1);
	}
	
	/**
	 * Liefert den Dantenbanknamen der Datenbank in der sich das ursprngliche 
	 * lotus.domino.Document befand.
	 * @return
	 */
	public String getParentDatabaseFileName() {
		return parentDatabaseFileName;
	}

	/**
	 * Liefert den Dantenbankdateipfad der Datenbank in der sich das ursprngliche 
	 * lotus.domino.Document befand.
	 * @return
	 */
	public String getParentDatabaseFilePath() {
		return parentDatabaseFilePath;
	}

	/**
	 * Liefert die Datenbank in der sich das ursprngliche 
	 * lotus.domino.Document befand.
	 * @return
	 */
	public Database getParentDatabase(Session session) throws NotesException {
		if (parentDatabaseReplicaID==null || parentDatabaseReplicaID.equals("")) {
			throw new NotesException(DJCACHEDOMINOEXCEPTIONID, "Das DJCacheDocument hat keine Parent Datenbank.");
		}
		return session.getDbDirectory("").openDatabaseByReplicaID(parentDatabaseReplicaID);
	}
	
	/**
	 * Ldt das Ursprngliche lotus.domino.Document aus der Datenbank
	 * @param session
	 * @return
	 * @throws NotesException
	 */
	public Document getDocument (Session session) throws NotesException {
		if (universalID == null) {
			throw new NotesException(DJCACHEDOMINOEXCEPTIONID, "Das DJCacheDocument hat keine Universal ID.");
		}
		return getParentDatabase(session).getDocumentByUNID(universalID);
	}

	/**
	 * Erzeugt ein neues lotus.domino.Document mit den Items aus dem DJCacheDocuemnt.
	 * Das Document wird als Speicher Instanz zurckgegeben, ohne gespeichert zu sein.
	 * Besondere Eigenschaften, wie isAuthors oder isReaders in den einzelnen
	 * Items gehen verloren, da Sie zur Zeit nicht von appendItemValue untersttzt werden.
	 * Untersttzt werden die ItemTypen Text, Number, DateTime und Vectoren davon.
	 * Ursprngliche Autoren- oder Leserfelder werden als normale Textfelder zurckgegeben.
	 * @param db
	 * @return
	 * @throws NotesException
	 */
	public Document createDocument (Database db) throws NotesException {
		Document result = db.createDocument();
		Vector v = this.getAllItemNames();
		for (int i=0, s=v.size(); i<s; i++) {
			String name=(String)v.elementAt(i);
			if (this.getType(name) != null && this.getType(name).equals(DateTime.class.getName())) {
				result.replaceItemValue (name,this.getItemValueDateTimeArray(name,db.getParent()));
			} else {
				result.replaceItemValue(name,this.getItemValue(name));
			}
		}
		return result;
	}
	
	/**
	 * Testet den Feldnamen auf Gltigkeit
	 * @param fieldName
	 * @return
	 */
	private static boolean ok (String fieldName) {
		return fieldName != null && fieldName.length() > 0;
	}
	
	
	/* ***************************************************************
	 * ***** HILFS-Methoden ... **************************************
	 *****************************************************************/
	
	/*
	 * Initialisierung einiger Variablen und bertragung des lotus.domino.Document
	 * in das DJCacheDocument
	 */
	private void init (Document doc) throws NotesException {
		universalID = doc.getUniversalID();
		parentDocumentUNID = doc.getParentDocumentUNID();
		parentDatabaseFilePath = doc.getParentDatabase().getFilePath();
		parentDatabaseFileName = doc.getParentDatabase().getFileName();
		parentDatabaseReplicaID = doc.getParentDatabase().getReplicaID();

		Vector v = doc.getItems();
		Item item = null;
		/* Da getItems bei Items mit gleichem Namen die zuletzt erzeugeten Items als erstes zurckgibt
		 * muss das DJCacheDocument mit den letzen zuerst gefllt werden, damit die Reihenfolge stimmt. */
		for (int i = v.size() - 1; i >= 0; i--) {
			item = (Item) v.elementAt(i);
			if (item != null
				&& (item.getType() == Item.TEXT
					|| item.getType() == Item.AUTHORS
					|| item.getType() == Item.DATETIMES
					|| item.getType() == Item.NAMES
					|| item.getType() == Item.NUMBERS)) {
				appendItemValue(item.getName(), item.getValues());
			}
		}
	}
	
	/*
	 * Wandelt potentielle DateTime Elemente in einem Vector in java.util.Date um.
	 * Nicht DateTime Elemente werden ignoriert
	 * @see isJavaDate.
	 * @param source
	 * @return
	 */
	protected static Vector toJavaDate (Vector source) {
		if (source==null) {
			return null;
		}
		Vector newVector = new Vector();
		for (int i = 0; i<source.size(); i++) {
			try {
				newVector.add(((DateTime) source.elementAt (i)).toJavaDate());
			} catch (Exception e) {
				//Nicht DateTime wird ignoriert -> siehe isJavaDate.
			}
		}
		return newVector;
	}
	
	/*
	 * Wandelt potentielle Integer Elemente in einem Vector in Double Elemente um.
	 * Nicht Integer Elemente werden ignoriert
	 * @see isInteger.
	 * @param source
	 * @return
	 */
	protected static Vector toDouble (Vector source) {
		if (source==null) {
			return null;
		}
		Vector newVector = new Vector();
		for (int i = 0; i<source.size(); i++) {
			try {
				newVector.add(new Double (((Integer) source.elementAt (i)).doubleValue()));
			} catch (Exception e) {
				//Nicht Integer wird ignoriert -> siehe isInteger.
			}
		}
		return newVector;
	}
	
	/*
	 * Wenn mindestens ein Element des Vectors vom Typ DateTime ist, return true
	 * @param source
	 * @return
	 */
	protected static boolean isDateTime (Vector source) {
		if (source==null) {
			return false;
		}
		for (int i = 0; i<source.size(); i++) {
			if (source.elementAt(i) instanceof DateTime) {
				return true;
			}
		}
		return false;
	}
	
	/*
	 * Wenn mindestens ein Element des Vectors vom Typ Integer ist, return true
	 * @param source
	 * @return
	 */
	protected static boolean isInteger (Vector source) {
		if (source==null) {
			return false;
		}
		for (int i = 0; i<source.size(); i++) {
			if (source.elementAt(i) instanceof Integer) {
				return true;
			}
		}
		return false;
	}
	
	/*
	 * Die Klasse FieldNames wird verwendet, um doppelte Feldnamen zu verwalten.
	 * Diese Fhigkeit wird bentigt, da die Methode appendItemValue es erlaubt,
	 * in mehreren Items gleichen Namens Werte zu speichern.
	 * Feldnamen unterscheiden nicht nach Gro- und Kleinschreibung 
	 * und werden kleingeschrieben gespeichert.
	 * Ein Feldname kann entweder aus einem Element (name) oder aus mehreren 
	 * (name, name!1 bis name!n) bestehen.
	 * Die Klasse FieldNames verwaltet beliebig viele solcher Feldnamen.
	 */
	protected class FieldNames {
		protected static final char SIGN = '!';
		protected Set names = Collections.synchronizedSet(new HashSet());
		protected Map types = Collections.synchronizedMap(new HashMap ());

		/**
		 * prft, ob es zu fieldName einen (ersten) Eintrag gibt.
		 * fieldName wird immer mit lowerCase verwendet
		 * @param fieldName
		 * @return
		 */
		protected boolean hasName (String fieldName) {
			if (!ok (fieldName)) { return false; }
			return (names.contains(makeName (fieldName)));
		}

		/**
		 * prft, ob es zu fieldName den i-ten Eintrag gibt
		 * fieldName wird immer mit lowerCase verwendet
		 * @param fieldName
		 * @param i
		 * @return
		 */
		protected boolean hasName (String fieldName, int i) {
			if (!ok (fieldName)) { return false; }
			return (names.contains(makeName (fieldName, i)));
		}
		
		/**
		 * erzeugt den Basisnamen zu fieldName, der spter als Key verwendet werden kann.
		 * Fhrt KEINE berprfung durch, ob es den Namenseintrag auch gibt.
		 * Fhrt KEINE berprfung durch, ob der Feldname zulssig ist.
		 * @see hasName (String)
		 * @param fieldName
		 * @return
		 */
		protected String makeName (String fieldName) {
			return fieldName.toLowerCase();
		}
		
		/**
		 * erzeugt den i-ten Namen zu fieldName, der spter als Key verwendet werden kann.
		 * Fhrt KEINE berprfung durch, ob es den Namenseintrag auch gibt.
		 * Fhrt KEINE berprfung durch, ob der Feldname zulssig ist.
		 * @see hasName (String, int)
		 * @param fieldName
		 * @param i
		 * @return
		 */
		protected String makeName (String fieldName, int i) {
			if (i==0) { return fieldName.toLowerCase(); }
			return new StringBuffer (fieldName.toLowerCase()).append(SIGN).append(i).toString();
		}
		
		/**
		 * Gibt einen Vector aller tatschlich als Eintrag gefundenen Namen zurck.
		 * @param fieldName
		 * @return
		 */
		protected Vector getNames (String fieldName) {
			if (!hasName (fieldName)) {//prft implizit auch ob fieldName != null
				return null;
			}
			Vector v = new Vector();
			int i = 0;
			while (hasName (fieldName, i)) {
				v.add (makeName (fieldName,i));
				i++;
			}
			return v;
		}
		
		/**
		 * Anzahl der gefundenen Namen
		 * @param fieldName
		 * @return
		 */
		protected int getNameCount (String fieldName) {
			if (!hasName(fieldName)) {
				return 0;
			}
			return getNames(fieldName).size();
		}
		
		/**
		 * fgt einen Neuen Namen an. Liegt noch kein Name vor, wird der Name
		 * als fieldName ansonsten
		 * als fieldName!n angelegt, wobei n-1 Namen vorlagen.
		 * @param fieldName
		 */
		protected String addName (String fieldName) {
			if (!ok (fieldName)) {
				return null;
			}
			String name=null;
			if (!hasName (fieldName)) {
				name = makeName (fieldName);
			} else {
				name = makeName (fieldName, getNameCount(fieldName));
			}
			names.add(name);
			return name;
		}
		
		/**
		 * Entfernt alle Namen zu einem Schlssel
		 * @param fieldName
		 */
		protected void removeName (String fieldName) {
			if (!hasName (fieldName)) {
				return;
			}
			Vector v = getNames (fieldName);
			for (int i = 0; i < v.size(); i++) {
				names.remove(makeName (fieldName, i));
			}
		}
		
		/**
		 * Merkt sich zu einem fieldName die zugehrige Klasse eines Objekts.
		 * @param fieldName
		 * @param value
		 */
		protected void setType (String fieldName, String value) {
			setType (fieldName, value, 0);
		}

		protected void setType (String fieldName, String value, int i) {
			if (!ok (fieldName)) {
				return;
			}
			types.put(makeName(fieldName,i), value);
		}
		
		/**
		 * gibt einen durch setType gesetzten Typ zurck
		 * @param fieldName
		 * @return
		 */
		protected String getType (String fieldName) {
			return getType (fieldName, 0);
		}
		
		protected String getType (String fieldName, int i) {
			if (!ok (fieldName)) {
				return null;
			}
			return (String) types.get(makeName (fieldName,i));
		}
		
		/**
		 * entfernt einen durch setType gesetzten Typ
		 * @param fieldName
		 * @return
		 */
		protected void removeType (String fieldName) {
			removeType (fieldName, 0);
		}
		
		protected void removeType (String fieldName, int i) {
			if (!ok (fieldName)) {
				return;
			}
			types.remove(makeName (fieldName,i));
		}
		
		/**
		 * Lierert Alle BasisNamen als Vector
		 * Es werden keine (internen) Namen von doppelten Elementen zurckgegeben.
		 * @return
		 */
		protected Vector getAllNames () {
			Vector v = new Vector ();
			Iterator e = names.iterator();
			while (e.hasNext()) {
				String part = (String) e.next();
				int index = part.lastIndexOf(SIGN);
				part = index != -1?part.substring(0,index):part;
				if (!v.contains(part)) {
					v.add (part);
				}
			}
			return v;
		}
		
	}
}